-- WhoDAT Social Tracker - Fixed with Deduplication and Role Detection

local ADDON_NAME = "WhoDAT"
local NS = _G[ADDON_NAME] or {}

-- Helper function for table length
local function tablelength(T)
  local count = 0
  for _ in pairs(T) do count = count + 1 end
  return count
end

-- Wrath-compatible timer function (C_Timer doesn't exist in 3.3.5a)
local function CreateTimer(delay, callback)
  local frame = CreateFrame("Frame")
  local elapsed = 0
  frame:SetScript("OnUpdate", function(self, dt)
    elapsed = elapsed + dt
    if elapsed >= delay then
      frame:SetScript("OnUpdate", nil)
      callback()
    end
  end)
  return frame
end

-- ============================================================================
-- DEDUPLICATION SYSTEM
-- ============================================================================

local lastRecordedGroup = nil
local lastRecordTime = 0

-- Generate a unique signature for a group composition
local function GenerateGroupSignature(composition)
  if not composition or not composition.members then
    return nil
  end
  
  -- Sort members by name for consistent signature
  local memberNames = {}
  for _, member in ipairs(composition.members) do
    table.insert(memberNames, member.name)
  end
  table.sort(memberNames)
  
  -- Create signature: zone|instance|memberlist
  local instanceStr = ""
  if composition.instance then
    instanceStr = composition.instance.name .. "|" .. (composition.instance.difficulty or 0)
  end
  
  local signature = composition.zone .. "|" .. instanceStr .. "|" .. table.concat(memberNames, ",")
  return signature
end

-- Check if this group composition should be recorded (deduplication)
local function ShouldRecordGroup(composition, reason)
  if not composition then
    return false
  end
  
  local currentSignature = GenerateGroupSignature(composition)
  if not currentSignature then
    return false
  end
  
  local currentTime = time()
  
  -- Always record if no previous group recorded
  if not lastRecordedGroup then
    lastRecordedGroup = currentSignature
    lastRecordTime = currentTime
    return true
  end
  
  -- Only record if the group composition actually changed
  if currentSignature ~= lastRecordedGroup then
    lastRecordedGroup = currentSignature
    lastRecordTime = currentTime
    return true
  end
  
  -- Skip if same group composition (no time-based duplicates)
  return false
end

-- ============================================================================
-- ROLE DETECTION SYSTEM (WoW 3.3.5a Compatible)
-- ============================================================================

-- Determine role based on class and talent spec (for Wrath 3.3.5a)
local function DetermineRole(unit)
  if not UnitExists(unit) then
    return "NONE"
  end
  
  local _, class = UnitClass(unit)
  if not class then
    return "NONE"
  end
  
  -- In Wrath 3.3.5a, we need to use GetTalentTabInfo for role detection
  local function GetActiveSpec()
    local maxPoints = 0
    local activeTab = 0
    
    for tabIndex = 1, 3 do
      local _, _, pointsSpent = GetTalentTabInfo(tabIndex)
      if pointsSpent and pointsSpent > maxPoints then
        maxPoints = pointsSpent
        activeTab = tabIndex
      end
    end
    
    return activeTab
  end
  
  -- Only check talents for the player (can't inspect others easily in dungeons)
  local isPlayer = UnitIsUnit(unit, "player")
  local activeSpec = isPlayer and GetActiveSpec() or 0
  
  -- Role detection based on class and spec
  if class == "WARRIOR" then
    if isPlayer and activeSpec == 3 then -- Protection
      return "TANK"
    else
      return "MELEE"  -- Arms/Fury assumed DPS
    end
    
  elseif class == "PALADIN" then
    if isPlayer and activeSpec == 2 then -- Protection  
      return "TANK"
    elseif isPlayer and activeSpec == 1 then -- Holy
      return "HEALER"
    else
      return "MELEE"  -- Retribution assumed DPS
    end
    
  elseif class == "HUNTER" then
    return "RANGED"
    
  elseif class == "ROGUE" then
    return "MELEE"
    
  elseif class == "PRIEST" then
    if isPlayer and activeSpec == 3 then -- Shadow
      return "RANGED"
    else
      return "HEALER"  -- Discipline/Holy assumed healing
    end
    
  elseif class == "SHAMAN" then
    if isPlayer and activeSpec == 3 then -- Restoration
      return "HEALER"
    elseif isPlayer and activeSpec == 2 then -- Enhancement
      return "MELEE"
    else
      return "RANGED"  -- Elemental assumed ranged DPS
    end
    
  elseif class == "MAGE" then
    return "RANGED"
    
  elseif class == "WARLOCK" then
    return "RANGED"
    
  elseif class == "DRUID" then
    if isPlayer and activeSpec == 3 then -- Restoration
      return "HEALER"
    elseif isPlayer and activeSpec == 2 then -- Feral
      -- Could be tank or melee, hard to distinguish without more info
      return "MELEE"  -- Default to melee for feral
    else
      return "RANGED"  -- Balance assumed ranged DPS
    end
    
  elseif class == "DEATHKNIGHT" then
    if isPlayer and activeSpec == 1 then -- Blood (tanking tree in Wrath)
      return "TANK"
    else
      return "MELEE"  -- Frost/Unholy assumed DPS
    end
  end
  
  -- Fallback
  return "DPS"
end

-- Enhanced role detection with group context
local function DetectRoleInGroup(unit, groupSize)
  local baseRole = DetermineRole(unit)
  
  -- In small groups, try to be more specific about roles
  if groupSize <= 5 then
    local _, class = UnitClass(unit)
    
    -- Tank classes in 5-mans are likely tanking
    if class == "WARRIOR" or class == "PALADIN" or class == "DEATHKNIGHT" then
      if baseRole == "TANK" or baseRole == "MELEE" then
        return "TANK"
      end
    end
    
    -- Healing classes in 5-mans are likely healing  
    if class == "PRIEST" or class == "SHAMAN" or class == "DRUID" or class == "PALADIN" then
      if baseRole == "HEALER" then
        return "HEALER"
      end
    end
  end
  
  return baseRole
end

-- ============================================================================
-- WRATH 3.3.5a COMPATIBILITY FUNCTIONS
-- ============================================================================

local function GetNumGroupMembers_Fixed()
  if UnitInRaid("player") then
    -- In Wrath, GetNumRaidMembers() includes yourself
    return GetNumRaidMembers()
  else
    -- In Wrath, GetNumPartyMembers() does NOT include yourself
    local numParty = GetNumPartyMembers()
    if numParty > 0 then
      return numParty + 1  -- +1 to include yourself
    end
    return 1 -- Solo player should return 1, not 0
  end
end

local function IsInGroup_Fixed()
  return GetNumPartyMembers() > 0 or UnitInRaid("player")
end

local function IsInRaid_Fixed()
  return UnitInRaid("player") ~= nil
end

-- ============================================================================
-- IMPROVED GROUP SCANNER WITH ROLE DETECTION
-- ============================================================================

local function ScanGroupComposition_Fixed()
  local inRaid = IsInRaid_Fixed()
  local inParty = GetNumPartyMembers() > 0
  
  -- Only record if actually in a group (not solo)
  if not inRaid and not inParty then 
    return nil 
  end
  
  local members = {}
  local actualSize = 0
  
  if inRaid then
    -- RAID: raid1, raid2, ... raid40
    local numRaid = GetNumRaidMembers()
    
    for i = 1, numRaid do
      local unit = "raid" .. i
      if UnitExists(unit) then
        local name = UnitName(unit)
        local _, class = UnitClass(unit)
        local level = UnitLevel(unit)
        local guid = UnitGUID(unit)
        
        if name and name ~= "" then
          local role = DetectRoleInGroup(unit, numRaid)
          table.insert(members, {
            name = name,
            class = class or "UNKNOWN",
            role = role,
            level = level or 0,
            guid = guid,
          })
          actualSize = actualSize + 1
        end
      end
    end
    
  else
    -- PARTY: player + party1, party2, party3, party4
    local groupSize = GetNumPartyMembers() + 1
    
    -- Add yourself first
    local playerName = UnitName("player")
    local _, playerClass = UnitClass("player")
    local playerLevel = UnitLevel("player")
    local playerGUID = UnitGUID("player")
    
    if playerName then
      local playerRole = DetectRoleInGroup("player", groupSize)
      table.insert(members, {
        name = playerName,
        class = playerClass or "UNKNOWN", 
        role = playerRole,
        level = playerLevel or 0,
        guid = playerGUID,
      })
      actualSize = 1
    end
    
    -- Add party members
    for i = 1, 4 do -- party1 through party4
      local unit = "party" .. i
      if UnitExists(unit) then
        local name = UnitName(unit)
        local _, class = UnitClass(unit)
        local level = UnitLevel(unit)
        local guid = UnitGUID(unit)
        
        if name and name ~= "" then
          local role = DetectRoleInGroup(unit, groupSize)
          table.insert(members, {
            name = name,
            class = class or "UNKNOWN",
            role = role, 
            level = level or 0,
            guid = guid,
          })
          actualSize = actualSize + 1
        end
      end
    end
  end
  
  -- Must have at least 2 people to be a "group"
  if actualSize < 2 then
    return nil
  end
  
  -- Get location info
  local zone = GetRealZoneText() or GetZoneText() or "Unknown"
  local subzone = GetSubZoneText() or ""
  
  -- Get instance info
  local instance = nil
  local inInstance, instanceType = IsInInstance()
  if inInstance then
    local name, type, difficulty, difficultyName = GetInstanceInfo()
    instance = {
      name = name or "Unknown Instance",
      type = type or instanceType or "unknown",
      difficulty = difficulty or 0,
      difficultyName = difficultyName or "Normal",
    }
  end
  
  local composition = {
    ts = time(),
    type = inRaid and "raid" or "party", 
    size = actualSize,
    members = members,
    instance = instance,
    zone = zone,
    subzone = subzone,
  }
  
  return composition
end

-- ============================================================================  
-- SMART RECORDING FUNCTION WITH DEDUPLICATION
-- ============================================================================

local function ForceRecordGroup(reason)
  local composition = ScanGroupComposition_Fixed()
  
  if not composition then
    return false
  end
  
  -- Check if we should record this group (deduplication)
  if not ShouldRecordGroup(composition, reason) then
    return false  -- Skip duplicate
  end
  
  -- Get player key
  local key = NS.Utils and NS.Utils.GetPlayerKey and NS.Utils.GetPlayerKey()
    or (GetRealmName() .. ":" .. UnitName("player"))
  
  -- Initialize data structure
  WhoDatDB = WhoDatDB or {}
  WhoDatDB.characters = WhoDatDB.characters or {}
  WhoDatDB.characters[key] = WhoDatDB.characters[key] or {}
  WhoDatDB.characters[key].events = WhoDatDB.characters[key].events or {}
  WhoDatDB.characters[key].events.groups = WhoDatDB.characters[key].events.groups or {}
  
  -- Add metadata
  composition.record_reason = reason
  composition._action = "group_formed"
  composition._domain = "social"
  composition._ts = composition.ts
  composition._session_id = NS.SessionID or "unknown"
  
  -- Insert the record
  table.insert(WhoDatDB.characters[key].events.groups, composition)
  
  -- Limit storage
  local maxGroups = 500
  while #WhoDatDB.characters[key].events.groups > maxGroups do
    table.remove(WhoDatDB.characters[key].events.groups, 1)
  end
  
  -- Also emit to EventBus if available
  if NS.EventBus and NS.EventBus.Emit then
    NS.EventBus:Emit("social", "group_formed", composition)
  end
  
  return true
end

-- ============================================================================
-- CONTROLLED EVENT REGISTRATION (REDUCED DUPLICATION)
-- ============================================================================

local fixFrame = CreateFrame("Frame", "WhoDAT_SocialTrackerFix")

-- Register only the most important group events (not every possible one)
local events = {
  "PARTY_MEMBERS_CHANGED",  -- Primary party event
  "RAID_ROSTER_UPDATE",     -- Primary raid event  
  "ZONE_CHANGED_NEW_AREA",  -- Zone changes (dungeons)
  "PLAYER_ENTERING_WORLD",  -- Initial login / UI reload
}

local registeredEvents = {}
for _, event in ipairs(events) do
  local success = pcall(function()
    fixFrame:RegisterEvent(event)
    registeredEvents[event] = true
  end)
end

-- Event handler with throttling
local eventThrottle = {}
fixFrame:SetScript("OnEvent", function(self, event, ...)
  local now = time()
  
  -- Throttle events to prevent spam (minimum 5 seconds between same event type)
  if eventThrottle[event] and (now - eventThrottle[event]) < 5 then
    return
  end
  
  eventThrottle[event] = now
  
  -- Small delay to let WoW update group info, then attempt recording
  CreateTimer(2, function()
    local recorded = ForceRecordGroup("event_" .. event)
    if recorded then
      NS.Log("INFO", "Recorded group change due to " .. event)
    end
  end)
end)

-- ============================================================================
-- INITIALIZATION AND OVERRIDES
-- ============================================================================

-- Replace the broken functions in the namespace if they exist
if NS.Social_ScanGroup then
  NS.Social_ScanGroup = ScanGroupComposition_Fixed
  print("[WhoDAT] Overrode NS.Social_ScanGroup with fixed version")
end

-- Force an initial scan after 5 seconds (only once on startup)
CreateTimer(5, function()
  local recorded = ForceRecordGroup("addon_startup")
  if recorded then
    NS.Log("INFO", "Initial group scan completed")
  end
end)

-- ============================================================================
-- Commands for Testing
-- ============================================================================

SLASH_WDGROUPFIX1 = "/wdgroupfix" 
SlashCmdList["WDGROUPFIX"] = function(msg)
  msg = (msg or ""):lower()
  
  if msg == "scan" then
    print("=== Manual Group Scan ===")
    local group = ScanGroupComposition_Fixed()
    if group then
      print(string.format("Found: %s with %d members", group.type, group.size))
      for i, member in ipairs(group.members) do
        print(string.format("  %d. %s (%s %s, level %d)", i, member.name, member.role, member.class, member.level))
      end
    else
      print("No group found")
    end
    
  elseif msg == "record" then
    print("=== Force Record Group ===")
    local success = ForceRecordGroup("manual_command")
    if success then
      print("✅ Group recorded successfully")
    else
      print("âŒ No group to record (not in party/raid)")
    end
    
  elseif msg == "status" then
    print("=== Group Status ===")
    print("GetNumPartyMembers():", GetNumPartyMembers())
    print("GetNumRaidMembers():", GetNumRaidMembers()) 
    print("UnitInRaid('player'):", tostring(UnitInRaid("player")))
    print("IsInGroup_Fixed():", tostring(IsInGroup_Fixed()))
    print("IsInRaid_Fixed():", tostring(IsInRaid_Fixed()))
    print("GetNumGroupMembers_Fixed():", GetNumGroupMembers_Fixed())
    
  elseif msg == "data" then
    print("=== Stored Group Data ===")
    local key = NS.Utils and NS.Utils.GetPlayerKey and NS.Utils.GetPlayerKey()
      or (GetRealmName() .. ":" .. UnitName("player"))
    
    local groups = WhoDatDB and WhoDatDB.characters and WhoDatDB.characters[key] 
      and WhoDatDB.characters[key].events and WhoDatDB.characters[key].events.groups or {}
    
    print(string.format("Total groups stored: %d", #groups))
    
    -- Show last 3 groups
    for i = math.max(1, #groups - 2), #groups do
      local group = groups[i]
      print(string.format("  %d. %s (%d members) at %s - %s", 
        i, group.type, group.size, 
        date("%H:%M:%S", group.ts),
        group.record_reason or "unknown"))
    end
    
  else
    print("=== WhoDAT Group Fix Commands ===")
    print("/wdgroupfix scan   - Test group scanning")
    print("/wdgroupfix record - Force record current group")
    print("/wdgroupfix status - Show group detection status")
    print("/wdgroupfix data   - Show stored group data")
  end
end